-- --------------------------------------------------------
-- From Chapter 2:
-- --------------------------------------------------------

# Using the MySQL Client

SHOW DATABASES;
USE test;

# Creating new users

CREATE DATABASE alpacas;
CREATE DATABASE movies;

GRANT SELECT, INSERT, UPDATE, DELETE, CREATE, DROP, ALTER, INDEX ON alpacas.* TO 'llama'@'localhost' IDENTIFIED BY 'camel';

GRANT FILE ON *.* TO  'llama'@'localhost';
GRANT CREATE ROUTINE, ALTER ROUTINE, EXECUTE, CREATE VIEW, SHOW VIEW ON alpacas.* TO 'llama'@'localhost';

GRANT USAGE ON *.* TO 'webuser'@'%' IDENTIFIED BY 'BroWs1ng';
GRANT SELECT ON alpacas.* TO 'webuser'@'%';
GRANT SELECT ON movies.* TO 'webuser'@'%';

SHOW GRANTS FOR 'llama'@'localhost';

# Deleting and revoking users

SELECT User, Host, Password FROM mysql.user;

DROP USER ''@'ubuntu';
DROP USER ''@'localhost';

SET PASSWORD FOR 'root'@'ubuntu' = PASSWORD('newPassword');

-- --------------------------------------------------------
-- From Chapter 4:
-- --------------------------------------------------------

# Choosing a Storage Engine

SHOW ENGINES;

# Character Sets and Collations

SHOW CHARACTER SET;

SHOW COLLATION LIKE 'latin1%'

# Creating Databases

CREATE DATABASE accounting;

SHOW DATABASES;

# Creating Tables

USE accounting;

CREATE TABLE invoices (
invoice_id SMALLINT(4) UNSIGNED NOT NULL AUTO_INCREMENT,
client_id SMALLINT(3) UNSIGNED NOT NULL,
invoice_date TIMESTAMP NOT NULL,
invoice_amount DECIMAL(10,2) UNSIGNED NOT NULL,
invoice_description TINYTEXT NOT NULL,
date_invoice_paid DATE,
PRIMARY KEY (invoice_id),
INDEX (client_id),
INDEX (invoice_date),
INDEX (invoice_amount),
INDEX (date_invoice_paid)
);

CREATE TABLE clients (
client_id SMALLINT(3) UNSIGNED NOT NULL AUTO_INCREMENT,
client_name VARCHAR(40) NOT NULL,
client_street VARCHAR(80) NOT NULL,
client_city VARCHAR(30) NOT NULL,
client_state CHAR(2) NOT NULL,
client_zip MEDIUMINT(5) UNSIGNED ZEROFILL NOT NULL,
client_phone VARCHAR(14),
contact_name VARCHAR(40),
contact_email VARCHAR(60),
PRIMARY KEY (client_id),
INDEX (client_name)
);

CREATE TABLE expenses (
expense_id SMALLINT(4) UNSIGNED NOT NULL AUTO_INCREMENT,
expense_category_id TINYINT(3) UNSIGNED NOT NULL,
expense_amount DECIMAL(10,2) UNSIGNED NOT NULL,
expense_description TINYTEXT NOT NULL,
expense_date TIMESTAMP NOT NULL,
PRIMARY KEY (expense_id),
INDEX (expense_category_id),
INDEX (expense_amount),
INDEX (expense_date)
);

CREATE TABLE expense_categories (
expense_category_id TINYINT(3) UNSIGNED NOT NULL AUTO_INCREMENT,
expense_category VARCHAR(30) NOT NULL,
PRIMARY KEY (expense_category_id),
UNIQUE (expense_category)
);

SHOW TABLES;
SHOW COLUMNS FROM invoices;

# Modifying Tables

USE accounting;

ALTER TABLE clients
CHANGE COLUMN contact_name
contact_first_name VARCHAR(15);

ALTER TABLE clients
ADD COLUMN contact_last_name VARCHAR(25)
AFTER contact_first_name;

SHOW COLUMNS FROM clients;

-- --------------------------------------------------------
-- From Chapter 5:
-- --------------------------------------------------------

# Inserting Data

USE accounting;

INSERT INTO expense_categories (expense_category) VALUES ('Travel-Hotel');

INSERT INTO expense_categories VALUES (NULL,  'Travel-Airfare');

INSERT INTO expense_categories VALUES
(NULL,  'Books'), 
(NULL,  'Web Hosting'), 
(NULL,  'Computer Software');

# THIS NEXT ONE IS NOT IN THE BOOK!

INSERT INTO expense_categories (expense_category) VALUES 
('Magazine Subscriptions'),  
('Computer Hardware'), 
('Sand Paper'), 
('Drum Sanders'), 
('Plaster of Paris'), 
('Erasers'), 
('Sandals'), 
('Horseshoes'), 
('Drywall'), 
('Paper Clips'), 
('Eggs'), 
('Light Switches'), 
('Electricity'), 
('Print Cartridges'), 
('Picture Frames'), 
('Internet Access Fees'), 
('Pencils'), 
('Employee Benefits');


# Populate the clients table.

INSERT INTO clients VALUES 
(NULL,  'Acme Industries',  '100 Main Street',  
'Anytown',  'NY',  11111,  '(888) 555-1234',  
'Jane',  'Doe',  'Jane@acme_industries.com');

INSERT INTO clients (client_name,  contact_first_name,  contact_last_name) VALUES 
('Winesburg Press',  'Sherwood',  'Anderson');

INSERT INTO clients VALUES 
(NULL,  'Galt on the Hill',  '1000 Tuttle Drive',  
'Brazilia',  'IL',  60000,  NULL,  NULL,  
'doe',  'jimmy.johns@example.com');

# THESE NEXT ONES ARE NOT IN THE BOOK!

INSERT INTO clients (client_name,  contact_first_name,  contact_last_name) VALUES 
('ABC Noun',  'Clarissa',  'Jones');

INSERT INTO clients VALUES (NULL,  'The Enterprises',  '200 Main Street',  'Anytown',  'NY',  11111,  '(800) 555-5555',  'John',  'Doe',  'Johnny_D@theenterprises.com');

INSERT INTO clients VALUES (NULL,  'Another Client',  '4 Some Ave',  'Newark',  'NJ',  07100,  NULL,  'Art',  'Vanderlay',  NULL);

INSERT INTO clients VALUES (NULL,  'Nepotism Inc.',  '693 Grand Blvd.',  'Fictional',  'ST',  89756,  NULL,  'Joey',  'B.',  'info@web-address.edu');

INSERT INTO clients VALUES (NULL,  'Something Clever',  'Some Road',  'Some City',  'OK',  64165,  NULL,  'Daryl',  'Zero',  NULL);

INSERT INTO clients VALUES (NULL,  'MiddleEarth Software',  '3902 4th Street',  'Patterson',  'NJ',  07509,  NULL,  'Rebecca',  NULL,  NULL);

# Selecting Data

SELECT * FROM expense_categories;

SELECT client_id,  client_name FROM clients;

INSERT INTO expenses VALUES 
(NULL, 3, '19.99', 'Larry Ullman\'s "MySQL: Visual QuickStart Guide"', '2006-04-20');
INSERT INTO expenses VALUES 
(NULL, 1, '104.50', 'Palmer House Hotel,  Chicago', '2006-01-26');
INSERT INTO expenses VALUES 
(NULL, 2, '689.00', 'Flight to Chicago', '2006-01-26');
INSERT INTO expenses VALUES 
(NULL, 5, '99.99', 'Mmmm...software', '2006-01-12');

# THESE NEXT ONES ARE NOT IN THE BOOK!

INSERT INTO expenses VALUES 
(NULL, 2, '64.99', 'Flight from Chicago', '2006-04-20');
INSERT INTO expenses VALUES 
(NULL, 7, '64.50', 'Apple PowerBook', '0000-00-00');
INSERT INTO expenses VALUES 
(NULL, 10, '6464.00', 'Sculpting', '2006-01-26');
INSERT INTO expenses VALUES (NULL, 22, '67.94', 'Writing implements needed to fill in little bubbles on tests.', '2006-04-20');
INSERT INTO expenses VALUES 
(NULL, 14, '1.97', 'Reconstruction', '2006-04-20');
INSERT INTO expenses VALUES 
(NULL, 8, '3216.00', 'Sanding', '2006-01-26');
INSERT INTO expenses VALUES 
(NULL, 8, '9712.97', 'There\'s a lot of sanding to do.', '2006-01-26');
INSERT INTO expenses VALUES 
(NULL, 5, '312.64', 'Software upgrade.', NULL);
INSERT INTO expenses VALUES 
(NULL, 20, '25.00', 'Frame to display Jess\' diploma.', '2006-05-09');
INSERT INTO expenses VALUES 
(NULL, 21, '39.99', 'Monthly expense for cable modem Internet access.', '2006-05-09');
INSERT INTO expenses VALUES 
(NULL, 3, '19.99', 'Larry Ullman\'s "PHP for the World Wide Web: Visual QuickStart Guide".', '2006-05-09');
INSERT INTO expenses VALUES 
(NULL, 3, '29.99', 'Larry Ullman\'s "PHP Advanced for the World Wide Web: Visual QuickPro Guide"', '2006-05-24');
INSERT INTO expenses VALUES 
(NULL, 5, '129.00', 'Software to create visual representations of databases.', '2006-05-24');

# Populate the clients table.

INSERT INTO invoices VALUES (NULL,4,'2006-04-24','1902.34','Conjugation: verbs, nouns, adjectives.', NULL);
INSERT INTO invoices VALUES (NULL,4,'2008-07-20','942.00','Technical writing.', NULL);

# THESE NEXT ONES ARE NOT IN THE BOOK!

INSERT INTO invoices VALUES (NULL,2,'2007-07-20','54.25','Hand wringing', NULL);
INSERT INTO invoices VALUES (NULL,3,'2006-04-24','1.00','Miscellaneous Services', NULL);
INSERT INTO invoices VALUES (NULL,1,'2006-04-10','654.34','Work, work, work.', NULL);
INSERT INTO invoices VALUES (NULL,1,'2008-01-20','98754.00','Technical writing.', NULL);
INSERT INTO invoices VALUES (NULL,2,'2007-07-18','54.25','Pacing.', NULL);
INSERT INTO invoices VALUES (NULL,3,'2006-04-24','3210.84','Pondering', NULL);
INSERT INTO invoices VALUES (NULL,4,'2007-04-08','6.64','Shady dealings.', NULL);
INSERT INTO invoices VALUES (NULL,4,'2008-11-20','2.00','Brilliance.', NULL);
INSERT INTO invoices VALUES (NULL,5,'2001-07-03','9.96','Don\'t ask.', NULL);
INSERT INTO invoices VALUES (NULL,2,'2006-04-22','315.94','Miscellaneous Services', NULL);
INSERT INTO invoices VALUES (NULL,3,'2006-12-24','9751.94','Reading.', NULL);
INSERT INTO invoices VALUES (NULL,4,'2006-07-20','321.55','HTML, PHP, MySQL Web development.', NULL);
INSERT INTO invoices VALUES (NULL,6,'2001-07-27','2.55','Hand wringing', NULL);
INSERT INTO invoices VALUES (NULL,7,'2007-10-24','64.64','Miscellaneous Services', NULL);

# Using Conditionals

SELECT expense_description FROM expenses 
WHERE expense_category_id = 3;

SELECT invoice_id, invoice_amount, invoice_date 
FROM invoices 
WHERE invoice_date >= '2006-03-01';

SELECT * FROM expenses 
WHERE expense_date IS NULL;

# Using LIKE and NOT LIKE

SELECT * FROM clients WHERE contact_last_name='Doe'\G

SELECT * FROM clients WHERE last_name LIKE 'Doe'\G

SELECT client_name, contact_first_name, contact_last_name
FROM clients WHERE
contact_first_name NOT LIKE 'Jo%';

# Performing Joins

SELECT invoice_amount, invoice_date,
client_name FROM invoices, clients
WHERE invoices.client_id = clients.client_id;

SELECT expense_category, expense_amount, expense_date FROM expense_categories, expenses WHERE expense_categories. expense_category_id = expenses. expense_category_id;

SELECT client_name, invoice_id, invoice_amount, invoice_date, invoice_description FROM clients LEFT JOIN invoices USING (client_id);

SELECT client_name, invoice_id, invoice_amount, invoice_date, invoice_description FROM invoices LEFT JOIN clients USING (client_id);

# Sorting Query Results

SELECT * FROM expense_categories ORDER BY expense_category;

SELECT client_id, invoice_date, invoice_amount FROM invoices ORDER BY client_id ASC, invoice_date DESC;

# Limiting Query Results

SELECT * FROM invoices
ORDER BY invoice_date LIMIT 1\G

SELECT expense_amount, expense_description FROM expenses,
expense_categories WHERE expenses.expense_category_id =
expense_categories.expense_category_id AND
expense_category = 'Sand Paper'
ORDER BY expense_amount DESC LIMIT 2;

# Updating Data

SELECT client_id, client_name FROM clients WHERE client_phone IS NULL;

UPDATE clients SET client_phone = '(800) 123-4567' WHERE client_id = 2;

SELECT * FROM clients WHERE client_id=2 \G

# Deleting Data

SELECT * FROM expense_categories
ORDER BY expense_category ASC;

DELETE FROM expense_categories WHERE
expense_category_id IN (1, 2);

INSERT INTO expense_categories VALUES (NULL, 'Travel');

SELECT expense_category_id FROM
expense_categories WHERE
expense_category='Travel';

UPDATE expenses SET expense_category_id = 24
WHERE expense_category_id IN (1, 2);

-- --------------------------------------------------------
-- From Chapter 6:
-- --------------------------------------------------------

# Text Functions

USE accounting;

SELECT TRIM(client_name) FROM clients;

SELECT SUBSTRING(client_phone, 2, 3) FROM clients WHERE client_phone IS NOT NULL;

SELECT LENGTH(expense_category), expense_category FROM 
expense_categories ORDER BY 
LENGTH(expense_category) DESC 
LIMIT 1;

# Concatenation and Alias

SELECT client_name, CONCAT(client_street, ', ', client_city, ', ', client_state, ' ', client_zip) AS address FROM clients;

SELECT expense_amount, expense_date,
CONCAT(expense_category, ': ', expense_description) FROM expenses,
expense_categories WHERE
expenses.expense_category_id = expense_categories.expense_category_id;

SELECT invoices.*,
CONCAT(client_name, ' - ', clients.client_id) AS client_info
FROM invoices LEFT JOIN clients
USING (client_id)
ORDER BY invoice_amount DESC
LIMIT 3\G

SELECT expense_amount, expense_date, CONCAT(expense_category, ': ', expense_description) FROM
expenses AS e,
expense_categories AS e_c
WHERE e.expense_category_id = e_c.expense_category_id;

# Numeric Functions

SELECT *, CONCAT('$', FORMAT (invoice_amount, 2)) AS amount FROM invoices ORDER BY invoice_date ASC\G

SELECT ROUND(expense_amount), expense_amount FROM expenses;

SELECT client_id, client_name FROM clients ORDER BY RAND();
SELECT client_id, client_name FROM clients ORDER BY RAND();

# Date and Time Functions

SELECT * FROM invoices WHERE
MONTH(invoice_date) = 4\G

SELECT CURDATE(), CURTIME();

SELECT CONCAT('$', FORMAT(expense_amount, 2)) AS amount,
expense_date, expense_category,
expense_description
FROM expenses AS e,
expense_categories AS e_c
WHERE (e.expense_category_id = e_c.expense_category_id)
AND (expense_date BETWEEN
SUBDATE(CURDATE(), INTERVAL 2 MONTH)
AND CURDATE());

# Formatting the Date and Time

SELECT DATE_FORMAT(NOW(),'%M %e, %Y - %l:%i');

SELECT TIME_FORMAT(CURTIME(),'%T');

SELECT DATE_FORMAT(expense_date, '%a %b %e %Y') AS the_date,
CONCAT('$', FORMAT(expense_amount, 2)) AS amount
FROM expenses
ORDER BY expense_date ASC,
expense_amount DESC;

# Encryption Functions

CREATE TABLE logins (
login_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
client_id SMALLINT UNSIGNED NOT NULL,
login_name TINYBLOB,
login_pass CHAR(40),
PRIMARY KEY (login_id),
INDEX (client_id)
);

INSERT INTO logins
(client_id, login_name, login_pass) VALUES
(4, AES_ENCRYPT('larryeullman', 'w1cKet'), SHA1('larryPASS'));

SELECT client_id FROM logins WHERE (login_pass = SHA1('larryPASS') AND AES_DECRYPT(login_name, 'w1cKet') = 'larryeullman');

# Grouping Functions

SELECT MAX(invoice_amount) FROM invoices;

SELECT SUM(expense_amount), expense_category FROM expenses LEFT JOIN expense_categories USING (expense_category_id) GROUP BY (expenses.expense_category_id);

SELECT COUNT(*) AS num, client_name
FROM invoices LEFT JOIN clients
USING (client_id)
GROUP BY (clients.client_id)
ORDER BY num DESC;

SELECT COUNT(*), SUM(expense_amount),
expense_category FROM expenses LEFT JOIN expense_categories
USING (expense_category_id)
GROUP BY (expenses.expense_category_id);

# Other Functions

INSERT INTO logins
(client_id, login_name, login_pass) VALUES
(4, AES_ENCRYPT('homerjsimpson', 'w1cKet'), SHA1('D\'ohReMi'));

SELECT LAST_INSERT_ID();

SELECT DISTINCT(invoices.client_id),
client_name FROM invoices, clients
WHERE invoices.client_id = clients.client_id;

SELECT COUNT(DISTINCT(client_id))
FROM invoices;


-- --------------------------------------------------------
-- From Chapter 7:
-- --------------------------------------------------------

# NO SQL COMMANDS! All PHP SCRIPTS!

-- --------------------------------------------------------
-- From Chapter 8:
-- --------------------------------------------------------

# NO SQL COMMANDS! All PERL SCRIPTS!

-- --------------------------------------------------------
-- From Chapter 9:
-- --------------------------------------------------------

# NO SQL COMMANDS! All JAVA APPLICATIONS!

-- --------------------------------------------------------
-- From Chapter 10:
-- --------------------------------------------------------

# Performing Transactions

USE test;

CREATE TABLE accounts (
id INT UNSIGNED NOT NULL AUTO_INCREMENT,
first_name VARCHAR(20) NOT NULL,
last_name VARCHAR(40) NOT NULL,
balance DECIMAL(10,2) NOT NULL DEFAULT 0.0,
PRIMARY KEY (id),
INDEX (last_name)
) ENGINE=InnoDB;

INSERT INTO accounts VALUES 
(NULL, 'Sarah', 'Vowell', 5460.23), 
(NULL, 'David', 'Sedaris', 909325.24), 
(NULL, 'Kojo', 'Nnamdi', 892.00);

SELECT * FROM accounts;

USE test;

START TRANSACTION;

UPDATE accounts SET balance=(balance-100) WHERE id=2;

UPDATE accounts SET balance=(balance+100) WHERE id=1;

SELECT * FROM accounts;

ROLLBACK;

SELECT * FROM accounts;

START TRANSACTION;

UPDATE accounts SET balance=(balance-100) WHERE id=2;

UPDATE accounts SET balance=(balance+100) WHERE id=1;

COMMIT;
SELECT * FROM accounts;


# Full-Text Searching

USE accounting;

SELECT expense_description FROM expenses;
ALTER TABLE expenses ADD FULLTEXT (expense_description);
SHOW CREATE TABLE expenses \G

USE accounting;

SELECT expense_id, expense_description
FROM expenses WHERE 
MATCH (expense_description) AGAINST ('visual');

SELECT expense_id, expense_description, 
MATCH (expense_description) AGAINST ('visual guide') AS rel 
FROM expenses WHERE 
MATCH (expense_description) AGAINST ('visual guide') \G

USE accounting;


SELECT expense_id, expense_description, 
MATCH (expense_description) AGAINST ('+visual +guide' IN BOOLEAN MODE) AS rel 
FROM expenses WHERE 
MATCH (expense_description) AGAINST ('+visual +guide' IN BOOLEAN MODE) ORDER BY rel DESC \G

SELECT expense_id, expense_description, 
MATCH (expense_description) AGAINST ('+visual +guide >quickpro' IN BOOLEAN MODE) AS rel 
FROM expenses WHERE 
MATCH (expense_description) AGAINST ('+visual +guide >quickpro' IN BOOLEAN MODE) ORDER BY rel DESC \G


# Regular Expressions

USE accounting;

SELECT client_name, client_phone FROM clients WHERE client_phone
REGEXP '^(800)|(888)|(877)';


SELECT client_name, client_phone 
FROM clients WHERE client_phone
REGEXP '^[(]?8{1}(00|88|77)';


SELECT client_name, contact_first_name, contact_email FROM clients 
WHERE contact_email NOT REGEXP 
'^[[:alnum:]_\.]+@.*\.[[:alnum:]]{2,3}';


# User-Defined Variables

USE accounting;

CREATE TABLE sodium (
salt CHAR(6) NOT NULL
);
INSERT INTO sodium (salt) VALUES ('w1cKet');
SELECT @salt:=salt FROM sodium;
INSERT INTO logins (client_id, login_name, login_pass) VALUES (3, AES_ENCRYPT('Isabella', @salt), SHA1('atticus7'));
SELECT @salt:=salt FROM sodium;
SELECT client_id, AES_DECRYPT(login_name, @salt) AS name FROM logins;
SELECT @salt;


# Introducing Unions

USE test;

CREATE TABLE table1 (
an_id TINYINT UNSIGNED NOT NULL AUTO_INCREMENT,
a_string VARCHAR(10),
a_date DATE,
PRIMARY KEY (an_id)
);
INSERT INTO table1 (a_string, a_date) VALUES 
('cat', '2006-03-10'), ('dog', '2006-02-06'), ('mouse', '2006-02-24'); 
CREATE TABLE table2 (
this_id TINYINT UNSIGNED NOT NULL AUTO_INCREMENT,
this_string VARCHAR(10),
this_date DATE,
PRIMARY KEY (this_id)
) AUTO_INCREMENT=3;
INSERT INTO table2 (this_string, this_date) VALUES 
('verb', '2006-03-10'), ('noun', '2006-03-24'), ('cat', '2006-04-20'); 


-- --------------------------------------------------------
-- From Chapter 11:
-- --------------------------------------------------------

# Stored Routines

USE accounting;

SELECT client_id, AES_DECRYPT(login_name, 'w1cKet') 
FROM logins ORDER BY client_id;

DELIMITER $$

CREATE PROCEDURE add_login 
(cid INT, name VARCHAR(20), pass VARCHAR(20)) 
BEGIN
	INSERT INTO logins 
(client_id, login_name, login_pass) 
VALUES 
(cid, AES_ENCRYPT(name, 'w1cKet'), SHA1(pass));
END $$

DELIMITER ;

CALL add_login (9, 'whatever', 'mypass');

SELECT client_id, AES_DECRYPT(login_name, 'w1cKet') FROM logins ORDER BY client_id;

USE accounting;

DELIMITER $$

CREATE FUNCTION days_old (d DATE) RETURNS INT
BEGIN
	RETURN DATEDIFF(d, CURDATE());
END $$

DELIMITER ;

SELECT invoice_id, days_old(invoice_date) AS days_unpaid FROM invoices WHERE date_invoice_paid IS NULL ORDER BY days_unpaid ASC;

USE accounting;

DELIMITER $$

CREATE FUNCTION capitalize 
(str VARCHAR(30)) RETURNS VARCHAR(30) 
BEGIN
	DECLARE s1 CHAR;
	DECLARE s2 VARCHAR(29);
	SET s1 = UPPER(LEFT(str, 1));
	SET s2 = SUBSTR(LOWER(str) FROM 2);
	RETURN CONCAT(s1, s2);
END $$
DELIMITER ;

SELECT capitalize('shady');

SELECT capitalize(contact_first_name) AS fn, capitalize(contact_last_name) AS ln FROM clients;

USE accounting;

DELIMITER $$

CREATE FUNCTION is_overdue (d DATE) RETURNS BOOLEAN
BEGIN
	DECLARE num INT;
	SET num = DATEDIFF(d, CURDATE());
	IF num < -45 THEN RETURN TRUE;
	ELSE RETURN FALSE;
	END IF;
END $$

DELIMITER ;

SELECT invoice_id, invoice_amount, 
days_old(invoice_date) AS days 
FROM invoices WHERE date_invoice_paid IS NULL 
AND is_overdue(invoice_date); 

USE test;

CREATE TABLE random_integers (
	num INT UNSIGNED NOT NULL
);

DELIMITER $$

CREATE PROCEDURE populate (many INT)
BEGIN
	DECLARE i INT DEFAULT 1;
	WHILE i <= many DO
		INSERT INTO random_integers 
		VALUES (FLOOR(1 + RAND() * 100));
		SET i = i + 1;
	END WHILE;
END $$

DELIMITER ;

CALL populate (10);
CALL populate (5);

SELECT * FROM random_integers;

USE accounting;

DELIMITER $$

CREATE PROCEDURE check_login (name VARCHAR(20), pass VARCHAR(20), OUT cid INT) 
BEGIN
	SELECT client_id INTO cid FROM logins WHERE login_name=AES_ENCRYPT(name, 'w1cKet') AND login_pass=SHA1(pass);
END $$

DELIMITER ;

CALL check_login ('username', 'password', @id);
SELECT @id;
CALL check_login ('bad', 'bad', @id);
SELECT @id;

# Triggers

CREATE DATABASE ecommerce;
USE ecommerce;

CREATE TABLE doodads (
doodad_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
doodad_name VARCHAR(40) NOT NULL,
doodad_price DECIMAL(10,2) UNSIGNED NOT NULL,
doodad_on_hand MEDIUMINT UNSIGNED NOT NULL DEFAULT 0,
PRIMARY KEY (doodad_id),
INDEX (doodad_name),
INDEX (doodad_price),
INDEX (doodad_on_hand)
);

CREATE TABLE orders (
order_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
doodad_id INT UNSIGNED NOT NULL,
PRIMARY KEY (order_id),
INDEX (doodad_id)
);

INSERT INTO doodads VALUES 
(NULL, 'a', 19.95, 20),
(NULL, 'b', 15.00, 10),
(NULL, 'c',  22.95, 5),
(NULL, 'd', 10.00, 15);

DELIMITER $$

CREATE TRIGGER update_qty_insert 
AFTER INSERT ON orders FOR EACH ROW 
BEGIN
	UPDATE doodads SET 
doodad_on_hand=doodad_on_hand-1 
WHERE doodads.doodad_id=NEW.doodad_id;
END $$

CREATE TRIGGER update_qty_delete 
BEFORE DELETE ON orders FOR EACH ROW 
BEGIN
	UPDATE doodads SET 
doodad_on_hand=doodad_on_hand+1 
WHERE doodads.doodad_id=OLD.doodad_id;
END $$

DELIMITER ;

INSERT INTO orders (doodad_id) 
VALUES (1), (2), (3), (1), (1);

SELECT * FROM doodads;

DELETE FROM orders WHERE order_id=4;

SELECT * FROM doodads;

# Views

USE accounting;

SELECT expense_category AS category, 
SUM(expense_amount) AS total 
FROM expense_categories AS ec 
LEFT JOIN expenses 
USING (expense_category_id) 
GROUP BY ec.expense_category_id

CREATE VIEW total_expenses AS 
SELECT expense_category AS category, 
SUM(expense_amount) AS total 
FROM expense_categories AS ec 
LEFT JOIN expenses 
USING (expense_category_id) 
GROUP BY ec.expense_category_id;

CREATE VIEW outstanding_invoices AS 
SELECT client_name AS client,
SUM(invoice_amount) AS total,
COUNT(invoice_id) AS invoices
FROM invoices JOIN clients USING (client_id) 
WHERE date_invoice_paid IS NULL 
AND is_overdue(invoice_date) 
GROUP BY invoices.client_id;

SHOW TABLES;

USE accounting;

SELECT category FROM total_expenses
WHERE total IS NULL;

SELECT * FROM outstanding_invoices;

SELECT invoice_id, invoice_amount, client_name
FROM clients JOIN invoices 
USING (client_id) 
WHERE is_overdue(invoice_date);
UPDATE invoices 
SET date_invoice_paid=NOW()
WHERE invoice_id IN (11, 15);

SELECT * FROM outstanding_invoices;

-- --------------------------------------------------------
-- From Chapter 12:
-- --------------------------------------------------------

# Storing Binary Data

CREATE DATABASE binary_data;
USE binary_data;

CREATE TABLE images (
image_id int(10) UNSIGNED NOT NULL AUTO_INCREMENT,
image BLOB,
image_name VARCHAR(60) NOT NULL,
image_type VARCHAR(12) NOT NULL,
image_width MEDIUMINT UNSIGNED NOT NULL,
image_height MEDIUMINT UNSIGNED NOT NULL,
image_size INT UNSIGNED NOT NULL,
uploaded_date TIMESTAMP,
PRIMARY KEY (image_id),
INDEX (uploaded_date)
);

# Making Query Result Pages

CREATE DATABASE movies;
USE movies;

CREATE TABLE directors (
director_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
first_name VARCHAR(20),
last_name VARCHAR(30) NOT NULL,
PRIMARY KEY (director_id),
INDEX (last_name)
);

CREATE TABLE films (
film_id INT UNSIGNED NOT NULL AUTO_INCREMENT,
director_id INT UNSIGNED NOT NULL,
title VARCHAR(100) NOT NULL,
year_released YEAR NOT NULL,
PRIMARY KEY (film_id),
INDEX (director_id),
INDEX (title),
INDEX (year_released)
);

INSERT INTO directors 
(first_name, last_name) 
VALUES 
('Wes', 'Anderson'),
('Joel', 'Coen'),
('Paul Thomas', 'Anderson'),
('Zach', 'Braff'),
('Curtis', 'Hanson'),
('Marty', 'Scorsese'),
('Miranda', 'July'),
('Steven', 'Soderbergh');


INSERT INTO films
(director_id, title, year_released)
VALUES 
(1, 'The Royal Tenebaums', 2001),
(1, 'Bottle Rocket', 1996),
(1, 'Rushmore', 1998),
(2, 'Blood Simple', 1984),
(2, 'Raising Arizona', 1987),
(2, 'Miller\'s Crossing', 1990),
(2, 'Fargo', 1996),
(2, 'The Big Lebowski', 1998),
(2, 'O Brother, Where Art Thou?', 2000),
(2, 'The Man Who Wasn\'t There', 2001),
(2, 'Intolerable Cruelty', 2003),
(3, 'Boogie Nights', 1997),
(3, 'Magnolia', 2000),
(3, 'Punch-Drunk Love', 2002),
(4, 'Garden State', 2004),
(5, 'L.A. Confidential', 1997),
(5, 'Wonder Boys', 2000);

-- --------------------------------------------------------
-- From Chapter 13:
-- --------------------------------------------------------

# NO SQL COMMANDS! All MySQL ADMINISTRATION!
